//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	IWP 1
// Objective: 	Part 1 of IWP work - choice of child generation and imputation of parents' wealth at death
//
//==============================================================================

clear all

************************
*** Data import ********
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

drop if wlra_s == 0 //Responding person longitudinal weight - Balanced W1 to W19

merge m:1 xwaveid wave using individual_weight_vars_restricted.dta //individual wealth variables
drop if _merge == 2
drop _merge

xtset xwaveid wave

************************
*** Create variables ***
************************

*individual net worth
gen child_net_worth = pwassei - pwdebti // individual net worth for child generation

*household net worth
gen child_net_worth_HH = hwnwip - hwnwin // HH net worth for child generation

replace psyodf = 0 if psyodf == . | psyodf < 0
replace psyodm = 0 if psyodm == . | psyodm < 0
egen max_psyodf = max(psyodf), by(xwaveid) //father's year of death, spread across all waves
egen max_psyodm = max(psyodm), by(xwaveid) //mother's year of death, spread across all waves

gen year = wave + 2000

gen parents_died = 0 //indicator for how many parents will have died by end of year
forvalues x = 2001/2019{
	replace parents_died = 1 if year == `x' & (max_psyodf <= `x' & max_psyodf > 0 | max_psyodm <= `x' & max_psyodm > 0) //at least one parent will have died
	replace parents_died = 2 if year == `x' & max_psyodf <= `x' & max_psyodf > 0 & max_psyodm <= `x' & max_psyodm > 0 //both parents will have died (replaces 1s with 2s, hence 1 now means that only one parent will have died)
}

gen second_parent_died = 0 //indicator for whether second parent died in the year in question
replace second_parent_died = 1 if parents_died == 2 & L.parents_died < 2 

******************************
*** Choice of child cohort ***
******************************

gen second_parent_died_16 = 0 //indicator for whether second parent to die died in the year in question or in the past 16 years (17 year window)
replace second_parent_died_16 = 1 if parents_died == 2 & L16.parents_died < 2

egen second_parent_died_16_age = total(wlra_s) if wave == 17 & second_parent_died_16 == 1, by(hgage) //calculate proportion whose second parent to die died between 2001 and 2017 by single year of age
egen age_total = total(wlra_s) if wave == 17, by(hgage)
gen second_parent_died_16_age_share = second_parent_died_16_age*100/age_total

*Generate figure B.1 - if running these three lines, will need to run again from the start
//collapse second_parent_died_16_age_share [pweight = wlra_s], by(hgage)
//replace hgage = hgage + 1 //want age in 2018 for charting purposes 
//export delimited second_parent_died_share.csv, replace // figure B.1 

gen age_63_73 = 0 //indicator for aged 63-73 in 2017
replace age_63_73 = 1 if year - hgage <= 1954 & year - hgage >= 1944

egen A_age_63_73 = total(wlra_s) if second_parent_died_16 == 1 & wave == 17, by(age_63_73) //calculate aggregate share of people whose second parent died among the 63-73 cohort
egen B_age_63_73 = total(wlra_s) if wave == 17, by(age_63_73)
gen C_age_63_73 = A_age_63_73*100/B_age_63_73 
codebook C_age_63_73 //49% of these people had a second parent die over 2001-2017

**************************************
*** Analysis of inheritance receipt **
**************************************

*1. Total inheritance receipt - much lower than expected 
gen bequest_flag_2 = oifinh + F.oifinh //inheritance
gen bequest_flag_3 = oifinh + F.oifinh + F2.oifinh

codebook bequest_flag_2 if second_parent_died == 1 & age_63_73 == 1 & wave <= 17 // (206 + 20) / 655 = 35% received an inheritance
codebook bequest_flag_3 if second_parent_died == 1 & age_63_73 == 1 & wave <= 17 // (234 + 42 + 4) / 655 = 43% received an inheritance

*2. What share of these are from the first parent dying?

gen first_parent_died = 0 //flag for only first parent dying in year in question
replace first_parent_died = 1 if parents_died == 1 & L.parents_died < 1 

gen only_second_parent_died = 0 //flag for (only) second parent dying in year in question
replace only_second_parent_died = 1 if parents_died == 2 & L.parents_died == 1 

codebook bequest_flag_2 if only_second_parent_died == 1 & L.first_parent_died == 0 & L2.first_parent_died == 0 & age_63_73 == 1 & wave <= 17 // (164 + 16) / 547 = 33% - share similar to that above, suggests very few inheritances from first parent dying.
codebook bequest_flag_3 if only_second_parent_died == 1 & L.first_parent_died == 0 & L2.first_parent_died == 0 & age_63_73 == 1 & wave <= 17 // (189 + 35 + 4) / 547 = 42% - share similar to that above, suggests very few inheritances from first parent dying.

codebook bequest_flag_2 if first_parent_died == 1 & F.second_parent_died == 0 & F2.second_parent_died == 0 & age_63_73 == 1 & wave <= 17 // 10 / 233 = 4% received an inheritance, ditto
codebook bequest_flag_3 if first_parent_died == 1 & F.second_parent_died == 0 & F2.second_parent_died == 0 & age_63_73 == 1 & wave <= 17 // (12 + 1) / 233 = 6% received an inheritance, ditto

*3. What share of inheritances from the first parent dying occur when the parents had separated? - about half of those recorded above are from parents who had separated despite them comprising only 10% of the sample

gen parents_separated = 0
replace parents_separated = 1 if (fmnprea == 1 | fmnprea == 3 | fmpdiv == 1) & fmpjoin != 1 //flag for parents separated (if: (never married or divorced/separated when child aged 14 or divorced since then) and never reconciled)

codebook parents_separated if age_63_73 == 1 & wave == 17 // 130 / 1297 = 10%

codebook bequest_flag_2 if first_parent_died == 1 & F.second_parent_died == 0 & F2.second_parent_died == 0 & age_63_73 == 1 & wave <= 17 & parents_separated == 1 // 5 / 29 = 17% received an inheritance, 5/10 of those recorded above
codebook bequest_flag_3 if first_parent_died == 1 & F.second_parent_died == 0 & F2.second_parent_died == 0 & age_63_73 == 1 & wave <= 17 & parents_separated == 1 // (4 + 1) / 29 = 17% received an inheritance, 5/12 of those recorded above

*4. Can the second parent repartnering following the death of the first parent explain the low rates from 1.? - No

gen father_died = 0 //flag for father having died by the end of the year in question
gen mother_died = 0 //flag for mother having died by the end of the year in question
forvalues x = 2001/2019{
	replace father_died = 1 if year == `x' & max_psyodf < `x' & max_psyodf > 0
	replace mother_died = 1 if year == `x' & max_psyodm < `x' & max_psyodm > 0
}

gen mother_new_partner_int = 1 if father_died == 1 & psmlhp == 1 //flag for mother having a new partner in the year in question - father died prior to the end of the previous year and mother lives with a partner
gen father_new_partner_int = 1 if mother_died == 1 & psflhp == 1 //flag for father having a new partner in the year in question - mother died prior to the end of the previous year and father lives with a partner
egen mother_new_partner = max(mother_new_partner_int), by(xwaveid) //spread across all waves
egen father_new_partner = max(father_new_partner_int), by(xwaveid) //spread across all waves

codebook mother_new_partner if age_63_73 == 1 & wave == 17 //13 mothers repartnered
codebook father_new_partner if age_63_73 == 1 & wave == 17 //14 fathers repartnered - 27 / 1297 in total - only 2%, so not a driving factor

*5. What are the patterns of multiple inheritance receipt?

gen bequest = oifinha
replace bequest = 0 if bequest == -1
replace bequest = . if bequest < -1

gen bequest_ratio_0_1 = bequest / F.bequest
gen bequest_ratio_1_2 = F.bequest / F2.bequest
gen bequest_ratio_0_2 = bequest / F2.bequest

tab bequest_ratio_0_1 if second_parent_died == 1 & age_63_73 == 1 & wave <= 17,m //some clustering around 1
tab bequest_ratio_1_2 if second_parent_died == 1 & age_63_73 == 1 & wave <= 17,m //some clustering around 1
tab bequest_ratio_0_2 if second_parent_died == 1 & age_63_73 == 1 & wave <= 17,m //nothing close to 1

***************
*** Approach **
***************
*Based on analysis above:
**drop cases where parent has new partner following death of partner (about 2%)
**drop cases where parents separated prior to death (about 10%)
**use allow three years (inc current year) for inheritance to be received following death of second parent 
**where recipient reports receiving multiple inheritances over these three years, proceed as follows:
****if inheritances are of a similar size (within 10%), assume this is double counting due to incorrect recall, so take average inheritance size to be the actual inheritance
****if inheritances are not of a similar size, assume the inheritance arrived in several parts (eg. liquid assets first, sale of property second), so take the sum to be the actual inheritance
****drop if three inheritances received in three years, as this may be a trust arrangement that has been incorrectly characterised by the recipient as an inheritance

*********************************************
*** Impute second parent's wealth at death **
*********************************************

*Method and assumptions explained in section B.7

drop if parents_separated == 1
drop if father_new_partner == 1
drop if mother_new_partner == 1
egen max_bequest_flag_3 = max(bequest_flag_3), by(xwaveid) //spread across all waves
drop if max_bequest_flag_3 == 3

tab bsn wave,m //number of siblings
replace bsn = 0 if bsn == -1
replace bsn = . if bsn < -1
gen siblings_8_int = bsn if wave == 8 //only available in waves 8, 12 and 19. Impute using nearest value
gen siblings_12_int = bsn if wave == 12
gen siblings_19_int = bsn if wave == 19
egen siblings_8 = max(siblings_8_int), by(xwaveid)
egen siblings_12 = max(siblings_12_int), by(xwaveid)
egen siblings_19 = max(siblings_19_int), by(xwaveid)
gen siblings = siblings_8 if wave <=10
replace siblings = siblings_12 if wave >= 11 & wave <= 15
replace siblings = siblings_19 if wave >= 16
egen siblings_int = min(bsn), by(xwaveid)
replace siblings = siblings_int if siblings == .

gen F_bequest = F.bequest
replace F_bequest = 0 if F_bequest == .
gen F2_bequest = F2.bequest
replace F2_bequest = 0 if F2_bequest == .
gen F_siblings = F.siblings
replace F_siblings = siblings if F_siblings == .
gen F2_siblings = F2.siblings
replace F2_siblings = siblings if F2_siblings == .

gen parent_WAD_second = bequest * (siblings + 1) + F_bequest * (F_siblings + 1) + F2_bequest * (F2_siblings + 1) if second_parent_died == 1 //impute wealth at death of second parent
replace parent_WAD_second = (bequest * (siblings + 1) + F_bequest * (F.siblings + 1) + F2.bequest * (F2.siblings + 1))/2 if second_parent_died == 1 & bequest_flag_3 == 2 & ((bequest_ratio_0_1 > 0.9 & bequest_ratio_0_1 < 1.11) | (bequest_ratio_1_2 > 0.9 & bequest_ratio_1_2 < 1.11) | (bequest_ratio_0_2 > 0.9 & bequest_ratio_0_2 < 1.11)) //see 'APPROACH' section above
replace parent_WAD_second = 0 if parent_WAD_second == . & second_parent_died == 1

gen parent_WAD_first = bequest * (siblings + 1) + F_bequest * (F_siblings + 1) + F2_bequest * (F2_siblings + 1) if first_parent_died == 1 //impute wealth at death of first parent
replace parent_WAD_first = (bequest * (siblings + 1) + F_bequest * (F.siblings + 1) + F2.bequest * (F2.siblings + 1))/2 if first_parent_died == 1 & bequest_flag_3 == 2 & ((bequest_ratio_0_1 > 0.9 & bequest_ratio_0_1 < 1.11) | (bequest_ratio_1_2 > 0.9 & bequest_ratio_1_2 < 1.11) | (bequest_ratio_0_2 > 0.9 & bequest_ratio_0_2 < 1.11)) //see 'APPROACH' section above
replace parent_WAD_first = 0 if parent_WAD_first == . & first_parent_died == 1

egen parent_WAD_first_spread = max(parent_WAD_first), by(xwaveid)
replace parent_WAD_first_spread = 0 if parent_WAD_first_spread == .
gen parent_WAD = parent_WAD_second + parent_WAD_first_spread // this is imputed parent wealth at death. Note that it is defined only in the year that the second parent died


********************************************************************************
*** Generate parent characteristic variables to model parent's wealth at death *
********************************************************************************

*Note that these variables are matched to thew information collected about the responding person

egen yob_male = max(psyobf), by(xwaveid) //mother's and father's year of birth
egen yob_female = max(psyobm), by(xwaveid)
replace yob_male = . if yob_male == . | yob_male < 0
replace yob_female = . if yob_female == . | yob_female < 0
gen yob_male_squ = yob_male^2
gen yob_female_squ = yob_female^2

egen max_fmfsch = max(fmfsch), by(xwaveid) //mother's and father's 	highest schooling
egen max_fmmsch = max(fmmsch), by(xwaveid)
gen school_female = 1 if max_fmmsch == 5 //year 12
replace school_female = 2 if max_fmmsch == 4 //year 11
replace school_female = 3 if max_fmmsch == 3 //highschool, but max of year 10
replace school_female = 4 if max_fmmsch == 2 | max_fmmsch == 1 //none or primary school only
gen school_male = 1 if max_fmfsch == 5 //year 12
replace school_male = 2 if max_fmfsch == 4 //year 11
replace school_male = 3 if max_fmfsch == 3 //highschool, but max of year 10
replace school_male = 4 if max_fmfsch == 2 | max_fmfsch == 1 //none or primary school only

gen university_female_int = .
replace university_female_int = 1 if fmmhlq == 1 //went to university
replace university_female_int = 0 if (fmmhlq >= 2 & fmmhlq <= 8) | fmmhlq == -1 //didn't go to university
gen university_male_int = .
replace university_male_int = 1 if fmfhlq == 1 //went to university
replace university_male_int = 0 if (fmfhlq >= 2 & fmfhlq <= 8) | fmfhlq == -1 //didn't go to university
egen university_female = max(university_female_int), by(xwaveid) //indicator for whether mother went to university
egen university_male = max(university_male_int), by(xwaveid) //indicator for whether father went to university

gen birth_country_female = 3 //mother's country of birth (3 is other)
replace birth_country_female = 1 if fmmcob == 1101 //Australia
replace birth_country_female = 2 if fmmcob == 2100 | fmmcob == 1201 | fmmcob == 8102 | fmmcob == 8104  | fmmcob == 152 | fmmcob == 9225 //main English speaking countries (United Kingdom, New Zealand, Canada, USA, Ireland and South Africa).
replace birth_country_female = . if fmmcob < 0

gen birth_country_male = 3 //father's country of birth (3 is other)
replace birth_country_male = 1 if fmfcob == 1101 //Australia
replace birth_country_male = 2 if fmfcob == 2100 | fmfcob == 1201 | fmfcob == 8102 | fmfcob == 8104  | fmfcob == 152 | fmfcob == 9225 //main English speaking countries (United Kingdom, New Zealand, Canada, USA, Ireland and South Africa).
replace birth_country_male = . if fmfcob < 0

*****************************************
*** Additional filtering and cleaning ***
*****************************************

drop if second_parent_died != 1 //observations where second parent didn't die

tab psfkm if age_63_73 == 1 & (max_psyodf > wave + 2000 | max_psyodf == 0) //observations where second parent lived abroad
tab psmkm if age_63_73 == 1 & (max_psyodm > wave + 2000 | max_psyodm == 0)
tab psbkm if age_63_73 == 1 & (max_psyodm > wave + 2000 | max_psyodm == 0) & (max_psyodf > wave + 2000 | max_psyodf == 0)
egen max_psfkm = max(psfkm), by(xwaveid)
egen max_psmkm = max(psmkm), by(xwaveid)
egen max_psbkm = max(psbkm), by(xwaveid)
gen second_parent_overseas = 1 if max_psyodm == year & (max_psmkm == 9 | max_psbkm == 9)
replace second_parent_overseas = 1 if max_psyodf == year & (max_psfkm == 9 | max_psbkm == 9)
drop if second_parent_overseas == 1

drop if wave > 17 //drop wave > 17, as not enough time for the inheritance to come through for this group (hence, focus is on waves 1-17)

drop if age_63_73 != 1

rename age_63_73 child_63_73

rename yob child_yob

keep xwaveid parent_WAD yob_male yob_female yob_male_squ yob_female_squ school_male school_female university_male university_female birth_country_male birth_country_female siblings wave child_63_73 wlra_s child_net_worth child_net_worth_HH child_yob

save IWP_data_1.dta, replace


